// ===== helpers
const $ = (s) => document.querySelector(s);
const $$ = (s) => document.querySelectorAll(s);

function toast(msg){
  let t = $("#toast");
  if(!t){
    t = document.createElement("div");
    t.id = "toast";
    t.className = "toast";
    document.body.appendChild(t);
  }
  t.textContent = msg;
  t.classList.add("show");
  clearTimeout(toast._t);
  toast._t = setTimeout(()=> t.classList.remove("show"), 1100);
}

// ===== active nav
(function(){
  const file = (location.pathname.split("/").pop() || "index.html").toLowerCase();
  $$(".nav a").forEach(a=>{
    const href = (a.getAttribute("href") || "").toLowerCase();
    if(href === file) a.classList.add("active");
  });
})();

// ===== i18n dictionary
const I18N = {
  hr: {
    nav_writeups: "writeups",
    nav_projects: "projekti",
    nav_contact: "kontakt",

    i_hello: "Petar Miketa",
    i_student: "Student kibernetičke sigurnosti",

    idx_focus_t: "Što radim",
    idx_f1_t: "Sigurnost Web Stranica",
    idx_f1_d: "Testiranje sigurnosti inputa, autentikacije i sesija — fokus na zaštitu podataka i sprječavanje kompromitacije.",
    idx_f1_b1: "auth / sessions",
    idx_f1_b2: "data exposure",

    idx_f2_t: "Pomoć Kod Online Prevara",
    idx_f2_d: "Prepoznavanje obrazaca online prevara, savjeti za izbjegavanje i pomoć oko daljnih koraka ako se prevara već desila.",
    idx_f2_b1: "phishing",
    idx_f2_b2: "soc-eng",

    idx_f3_t: "Praćenje i Analiza Internet Prometa",
    idx_f3_d: "Analiza prometa i anomalija — detekcija sumnjivog ponašanja i tragova napada web stranica i web aplikacija.",

    idx_f4_t: "Vlastiti Alati za Provjeru Sigurnosti",
    idx_f4_d: "Razvoj vlastitih programskih alata za provjeru sigurnosti sustava i sigurnosnih slabosti po potrebama projekta.",
    idx_f4_b2: "automation",

    idx_f5_t: "Testiranje Stabilnosti Sustava",
    idx_f5_d: "Namjerno i kontrolirano opterećenje granica sustava na: Programima, Aplikacijama i Web Stranicama.",
    idx_f5_b1: "rate-limit",
    idx_f5_b2: "stability",

    idx_stack_t: "Tech Stack",
    idx_stack1_t: "Sustavi",
    idx_stack2_t: "Code / mreže",
    idx_stack_win: "System Fundamentals & Troubleshooting",
    idx_stack_git: "Repo Workflow",

    prev_w_t: "Write-ups",
    prev_p_t: "Projects",
    prev_all: "Pogledaj sve",
    prev_w_ex1_t: "Phishing prevare putem lažnih oglasa",
    prev_w_ex1_d: "Pregled prevare + kako je prepoznati i izbjeći kroz jasne signale",
    btn_read_writeups: "Pogledaj write-upove",

    prev_p_ex1_t: "SSH Log Analyzer CLI",
    prev_p_ex1_d: "CLI alat za analizu SSH logova i detekciju brute-force napada",
    btn_view_projects: "Pogledaj projekte",

    w_title: "Write-ups",
    w_desc: "Analize scenarija s fokusom na tehničku pozadinu, napadački lanac i obrambene mjere. Sve je lab-only / edukativno.",
    w_ph: "Pretraži (SQLi, XSS, phishing, Wireshark)...",
    w_reset: "Očisti",

    p_title: "Projects",
    p_desc: "Stavi svoje projekte i linkove (uskoro više).",
    p_open_project: "Otvori projekt",
    p_open: "Otvori",
    p_soon: "(uskoro)",

    back_projects: "← Projekti",

    ssh_page_title: "SSH Log Analyzer CLI",
    ssh_page_sub: "Alat za analizu SSH autentikacijskih logova i detekciju potencijalnih brute-force napada.",
    ssh_quick_title: "Kako koristiti (brzo):",
    ssh_writeup_title: "Writeup",

    btn_github: "GitHub",
    btn_writeups: "Writeups",
    idx_stack2_k1: "Jezici",
    idx_stack2_k2: "Internet",
    idx_stack2_k3: "Sigurnost",

    c_title: "Kontakt",
    c_desc: "Za upite i suradnju — javite se putem forme ili preko linkova ispod.",
    c_reach: "Kontakt informacije",
    c_form_title: "Kontaktirajte me",
    c_form_desc: "Poruka se šalje direktno na moj email. Odgovaram u najkraćem mogućem roku.",
    c_name: "Ime",
    c_email: "Email",
    c_subject: "Tema",
    c_msg: "Poruka",
    c_send: "Pošalji",
    c_clear: "Očisti",

    c_ph_name: "Unesite vaše ime",
    c_ph_email: "Unesite vaš email",
    c_ph_subject: "Unesite temu poruke",
    c_ph_message: "Unesite poruku...",

    c_lbl_email: "Email",
    c_lbl_github: "GitHub",
    c_lbl_thm: "TryHackMe",
    c_lbl_ig: "Instagram",

    c_pgp_optional: "opcionalno",
    c_pgp_desc: "PGP je dodatna zaštita: poruku možete “zaključati” tako da je mogu pročitati samo ja. Ako vam je to previše komplicirano — pošaljite normalnu poruku, sve je okej.",
    c_pgp_where: "Gdje su upute?",
    c_pgp_how: "Nakon preuzimanja, u vašem <strong>Downloads</strong> folderu nalazi se ZIP paket. Unutra je <strong>README.txt</strong> s jednostavnim uputama kako koristiti šifriranje podataka.",
    c_pgp_download: "Preuzmi PGP paket (key + upute)",
    c_pgp_fpr: "Fingerprint (provjera ključa):",

    copied: "Kopirano ✓"
  },

  en: {
    nav_writeups: "writeups",
    nav_projects: "projects",
    nav_contact: "contact",

    i_hello: "Petar Miketa",
    i_student: "Cyber Security Student",


    idx_focus_t: "What I do",
    idx_f1_t: "Web Security",
    idx_f1_d: "Testing input handling, authentication, and sessions — focused on protecting data and preventing compromise.",
    idx_f1_b1: "auth / sessions",
    idx_f1_b2: "data exposure",

    idx_f2_t: "Help With Online Scams",
    idx_f2_d: "Spotting scam patterns, advice to avoid them, and next steps if a scam already happened.",
    idx_f2_b1: "phishing",
    idx_f2_b2: "soc-eng",

    idx_f3_t: "Network Traffic Monitoring & Analysis",
    idx_f3_d: "Traffic and anomaly analysis — detecting suspicious behavior and traces of attacks on websites and web apps.",

    idx_f4_t: "Custom Security Tools",
    idx_f4_d: "Building small tools/scripts to test and validate security weaknesses based on project needs.",
    idx_f4_b2: "automation",

    idx_f5_t: "System Stability Testing",
    idx_f5_d: "Controlled stress testing to understand system limits for programs, apps, and websites.",
    idx_f5_b1: "rate-limit",
    idx_f5_b2: "stability",

    idx_stack_t: "Tech Stack",
    idx_stack1_t: "Systems",
    idx_stack2_t: "Code / networking",
    idx_stack_win: "System fundamentals & troubleshooting",
    idx_stack_git: "Repo workflow",

    idx_stack2_k1: "Code",
    idx_stack2_k2: "Network",
    idx_stack2_k3: "Security",

    prev_w_t: "Write-ups",
    prev_p_t: "Projects",
    prev_all: "View all",
    prev_w_ex1_t: "Phishing scams via fake listings",
    prev_w_ex1_d: "Attack flow + how to recognize and avoid it using clear signals",
    btn_read_writeups: "View write-ups",

    prev_p_ex1_t: "SSH Log Analyzer CLI",
    prev_p_ex1_d: "CLI tool for analyzing SSH logs and detecting brute-force attacks",
    btn_view_projects: "View projects",

    w_title: "Write-ups",
    w_desc: "Scenario analyses focused on technical background, attack chain, and defensive measures. Lab-only / educational.",
    w_ph: "Search (SQLi, XSS, phishing, Wireshark)...",
    w_reset: "Clear",

    p_title: "Projects",
    p_desc: "My projects and links (more soon).",
    p_open_project: "Open project",
    p_open: "Open",
    p_soon: "(coming soon)",

    back_projects: "← Projects",

    ssh_page_title: "SSH Log Analyzer CLI",
    ssh_page_sub: "Tool for analyzing SSH authentication logs and detecting potential brute-force attacks.",
    ssh_quick_title: "Quick usage:",
    ssh_writeup_title: "Writeup",

    btn_github: "GitHub",
    btn_writeups: "Writeups",

    c_title: "Contact",
    c_desc: "For questions and collaboration — reach out via the form or the links below.",
    c_reach: "Contact info",
    c_form_title: "Send a message",
    c_form_desc: "Your message is sent directly to my email. I reply as soon as possible.",
    c_name: "Name",
    c_email: "Email",
    c_subject: "Subject",
    c_msg: "Message",
    c_send: "Send",
    c_clear: "Clear",

    c_ph_name: "Enter your name",
    c_ph_email: "Enter your email",
    c_ph_subject: "Enter a subject",
    c_ph_message: "Type your message...",

    c_lbl_email: "Email",
    c_lbl_github: "GitHub",
    c_lbl_thm: "TryHackMe",
    c_lbl_ig: "Instagram",

    c_pgp_optional: "optional",
    c_pgp_desc: "PGP is an extra layer of protection: you can “lock” a message so only I can read it. If that’s too much hassle — just send a normal message, totally fine.",
    c_pgp_where: "Where are the instructions?",
    c_pgp_how: "After downloading, you’ll find a ZIP package in your <strong>Downloads</strong> folder. Inside is <strong>README.txt</strong> with simple instructions on how to use encryption.",
    c_pgp_download: "Download PGP package (key + instructions)",
    c_pgp_fpr: "Fingerprint (key check):",

    copied: "Copied ✓"
  }
};

function applyLang(lang){
  const dict = I18N[lang] || I18N.hr;
  document.documentElement.lang = lang;

  $$("[data-i18n]").forEach(el=>{
    const key = el.getAttribute("data-i18n");
    if(key && dict[key] != null) el.textContent = dict[key];
  });

  $$("[data-i18n-html]").forEach(el=>{
    const key = el.getAttribute("data-i18n-html");
    if(key && dict[key] != null) el.innerHTML = dict[key];
  });

  $$("[data-i18n-placeholder]").forEach(el=>{
    const key = el.getAttribute("data-i18n-placeholder");
    if(key && dict[key] != null) el.setAttribute("placeholder", dict[key]);
  });

  $$(".lang-btn").forEach(b=>{
    b.classList.toggle("active", b.dataset.lang === lang);
  });

  localStorage.setItem("lang", lang);
  window.dispatchEvent(new CustomEvent("langchange", { detail: { lang } }));
}

(function initLang(){
  const saved = localStorage.getItem("lang") || "hr";
  applyLang(saved);

  $$(".lang-btn").forEach(btn=>{
    btn.addEventListener("click", ()=>{
      const next = btn.dataset.lang || "hr";
      applyLang(next);
      toast(next === "hr" ? "Hrvatski" : "English");
    });
  });
})();

// ===== typing effect (index only)
(function typing(){
  const el = $("#typeLine");
  // Only run typing effect if explicitly enabled on the element.
  // This prevents extra lines in the hero terminal and avoids layout mismatch.
  if(!el || el.dataset.enableTyping !== "1") return;

  let timer = null;

  function getLines(){
    const lang = localStorage.getItem("lang") || "hr";
    const d = I18N[lang] || I18N.hr;
    return [d.i_typing1, d.i_typing2, d.i_typing3].filter(Boolean);
  }

  let i = 0, j = 0, del = false;

  function stop(){
    if(timer) clearTimeout(timer);
    timer = null;
  }

  function reset(){
    stop();
    i = 0; j = 0; del = false;
    el.textContent = "";
    tick();
  }

  function tick(){
    const lines = getLines();
    if(lines.length === 0) return;

    const full = lines[i];

    if(!del){
      j++;
      el.textContent = full.slice(0, j);
      if(j >= full.length){
        del = true;
        timer = setTimeout(tick, 1200);
        return;
      }
    } else {
      j--;
      el.textContent = full.slice(0, j);
      if(j <= 0){
        del = false;
        i = (i + 1) % lines.length;
      }
    }
    timer = setTimeout(tick, del ? 28 : 20);
  }

  window.addEventListener("langchange", reset);
  tick();
})();

// ===== copy email (contact) — safe (only if button exists)
(function copyEmail(){
  const btn = $("#copyEmail");
  if(!btn) return;

  btn.addEventListener("click", async ()=>{
    const email = btn.getAttribute("data-email") || "";
    const lang = localStorage.getItem("lang") || "hr";
    const d = I18N[lang] || I18N.hr;

    try{
      await navigator.clipboard.writeText(email);
      toast(d.copied || (lang === "hr" ? "Email kopiran ✓" : "Email copied ✓"));
    }catch{
      window.prompt("Copy email:", email);
    }
  });
})();

// ===== writeups search + reset button visibility
(function writeupSearch(){
  const list = $("#writeupList");
  const input = $("#searchWriteups");
  const reset = $("#resetWriteups");
  if(!list || !input || !reset) return;

  const doFilter = (q)=>{
    const query = (q || "").trim().toLowerCase();
    $$("#writeupList .item").forEach(item=>{
      const text = (item.innerText || "").toLowerCase();
      const tags = (item.getAttribute("data-tags") || "").toLowerCase();
      item.style.display = (text.includes(query) || tags.includes(query)) ? "" : "none";
    });
  };

  const toggleReset = ()=>{
    reset.classList.toggle("show", input.value.trim().length > 0);
  };

  const syncResetLabel = ()=>{
    const lang = localStorage.getItem("lang") || "hr";
    const d = I18N[lang] || I18N.hr;
    reset.textContent = d.w_reset || (lang === "hr" ? "Očisti" : "Clear");
  };

  input.addEventListener("input", e=>{
    doFilter(e.target.value);
    toggleReset();
  });

  reset.addEventListener("click", ()=>{
    input.value = "";
    doFilter("");
    toggleReset();
    const lang = localStorage.getItem("lang") || "hr";
    toast(lang === "hr" ? "Pretraga očišćena" : "Search cleared");
    input.focus();
  });

  window.addEventListener("langchange", syncResetLabel);

  syncResetLabel();
  toggleReset();
})();

// ===== mobile nav toggle (injected, no HTML edits needed)
(function initMobileNav(){
  const topbar = document.querySelector(".topbar");
  const nav = document.querySelector(".nav");
  if(!topbar || !nav) return;

  if(topbar.querySelector(".nav-toggle")) return;

  const btn = document.createElement("button");
  btn.type = "button";
  btn.className = "nav-toggle";
  btn.setAttribute("aria-label", "Toggle menu");
  btn.setAttribute("aria-expanded", "false");
  btn.innerHTML = "<span></span>";

  // Some pages use a different DOM structure where .nav is NOT a direct child of .topbar.
  // insertBefore would throw and stop the entire script (breaking matrix + other features).
  try{
    if(nav.parentElement === topbar){
      topbar.insertBefore(btn, nav);
    }else{
      // put the toggle inside the topbar (right side by CSS), without assuming structure
      topbar.appendChild(btn);
    }
  }catch(e){
    // Never allow this helper to crash the whole site.
    try{ topbar.appendChild(btn); }catch(_e){}
  }

  btn.addEventListener("click", ()=>{
    const open = document.body.classList.toggle("nav-open");
    btn.setAttribute("aria-expanded", open ? "true" : "false");
  });

  nav.addEventListener("click", (e)=>{
    if(e.target && e.target.matches("a")){
      document.body.classList.remove("nav-open");
      btn.setAttribute("aria-expanded", "false");
    }
  });
})();

// ===== Lottie Loader (always on: entry + refresh + nav) =====
(function loaderAlways(){
  if (window.__PM_LOADER_INIT__) return;
  window.__PM_LOADER_INIT__ = true;

  // Loader is removed/disabled to avoid blur and layout issues.
  // Keeping this stub prevents older cached HTML from throwing errors.
  return;

  const MIN_MS = 800;
  const show = () => document.documentElement.classList.add("is-loading");
  const hide = () => document.documentElement.classList.remove("is-loading");

  let anim = null;
  let hideTimer = 0;

  function initLottie(){
    const el = document.getElementById("lottieLoader");
    if(!el) return;
    if(typeof lottie === "undefined") return;

    // reset container to avoid duplicates
    if(anim) { try{ anim.destroy(); }catch{} anim = null; }
    el.innerHTML = "";

    anim = lottie.loadAnimation({
      container: el,
      renderer: "svg",
      loop: true,
      autoplay: true,
      path: "assets/loading.json"
    });
  }

  // 1) loader start time (shared across refresh/back/forward too)
  let startAt = Date.now();
  try{
    const t = Number(sessionStorage.getItem("loaderStartAt") || "");
    if(t) startAt = t;
    else sessionStorage.setItem("loaderStartAt", String(startAt));
  }catch(e){}

  // 2) Ensure visible immediately (in case head script missing somewhere)
  show();

  // 3) Init animation
  document.addEventListener("DOMContentLoaded", initLottie);

  // 4) Hide after MIN_MS total since startAt
  window.addEventListener("load", () => {
    if(hideTimer) clearTimeout(hideTimer);

    let wait = MIN_MS;
    try{
      const elapsed = Date.now() - startAt;
      wait = Math.max(0, MIN_MS - elapsed);
    }catch(e){}

    hideTimer = setTimeout(() => {
      hide();
      try{ sessionStorage.removeItem("loaderStartAt"); }catch(e){}
    }, wait);
  });

  // 5) Internal navigation: set new startAt and show instantly
  document.addEventListener("click", (e) => {
    const a = e.target.closest("a");
    if(!a) return;

    const href = a.getAttribute("href") || "";
    const target = a.getAttribute("target");

    if(!href || href.startsWith("#")) return;
    if(href.startsWith("mailto:") || href.startsWith("tel:")) return;
    if(a.hasAttribute("download")) return;
    if(target === "_blank") return;

    let url;
    try{
      url = new URL(href, window.location.href);
      if(url.origin !== window.location.origin) return;
    }catch{ return; }

    e.preventDefault();

    const now = Date.now();
    try{ sessionStorage.setItem("loaderStartAt", String(now)); }catch(e){}
    startAt = now;

    if(hideTimer) clearTimeout(hideTimer);
    show();

    window.location.href = url.href;
  });

  // 6) BFCache: kad se vratiš back/forward, ne želimo stuck loader
  window.addEventListener("pageshow", (ev) => {
    if(ev.persisted){
      if(hideTimer) clearTimeout(hideTimer);
      hide();
      try{ sessionStorage.removeItem("loaderStartAt"); }catch(e){}
    }
  });
})();
// ===== Matrix background effect (runs only if <canvas id="matrix"> exists)
// NOTE: wrapped in DOMContentLoaded so it always finds the canvas.
window.addEventListener("DOMContentLoaded", () => {
  const c = document.getElementById("matrix");
  if(!c) return;
  const ctx = c.getContext("2d");

  const glyphs = "アイウエオカキクケコ";
  let w, h, cols, drops, fontSize;

  // Use devicePixelRatio so it looks crisp and avoids "nothing visible" on some setups.
  function resize(){
    const dpr = Math.max(1, window.devicePixelRatio || 1);
    w = window.innerWidth;
    h = window.innerHeight;
    c.width  = Math.floor(w * dpr);
    c.height = Math.floor(h * dpr);
    c.style.width = w + "px";
    c.style.height = h + "px";
    ctx.setTransform(dpr, 0, 0, dpr, 0, 0);

    // Smaller fontSize => more columns => denser rain.
    fontSize = Math.max(10, Math.floor(w / 110));
    cols = Math.floor(w / fontSize);
    drops = Array(cols).fill(1);
    ctx.font = `${fontSize}px monospace`;
  }

  // Throttle to ~16 FPS for a calmer, heavier look.
  let last = 0;
  function step(t){
    if(t - last < 45){
      requestAnimationFrame(step);
      return;
    }
    last = t;

    // trail
    ctx.fillStyle = "rgba(0, 0, 0, 0.08)";
    ctx.fillRect(0, 0, w, h);

    ctx.fillStyle = "rgba(0, 255, 80, 0.55)";
    for(let i=0; i<drops.length; i++){
      const ch = glyphs[Math.floor(Math.random()*glyphs.length)];
      ctx.fillText(ch, i*fontSize, drops[i]*fontSize);

      if(drops[i]*fontSize > h && Math.random() > 0.985) drops[i] = 0;
      // slower fall
      drops[i] += 0.7;
    }
    requestAnimationFrame(step);
  }

  window.addEventListener("resize", resize);
  resize();
  requestAnimationFrame(step);
});
// topbar scroll toggle
window.addEventListener("DOMContentLoaded", () => {
  const topbar = document.querySelector(".topbar");
  if (!topbar) return;

  const onScroll = () => {
    topbar.classList.toggle("is-scrolled", window.scrollY > 10);
  };

  window.addEventListener("scroll", onScroll, { passive: true });
  onScroll();
});
